function [r,dl,Nd] = rectangle_shell_distribution_dipoles(a1, a2, b1, b2,...
    c1, c2, N, pos0)

% This function obtains the position of the dipoles resulting from
% discretizing a in a cubic net a parallelepiped shell in N punctual 
% dipoles.

%%%%%% INPUT ARGUMENTS %%%%%%
%%% ai,bi and ci: superior and inferior limits of the shell            %%%
%%% N:            number of dipoles inside the scatterer               %%%
%%% pos0:         position of the center of mass of the parallelepiped %%%

%%%%%% OUTUT ARGUMENTS %%%%%%
%%% r: (Nd,3) vector containing the 3D positions of the Nd dipoles  %%%
%%% dl:        lattice parametwr of the resulting net               %%%

%%% Organizing limits %%%
if(a1 > a2)
    
    aux = a2;
    a2  = a1;
    a1  = aux;
    
end

if(b1 > b2)
    
    aux = b2;
    b2  = b1;
    b1  = aux;
    
end

if(c1 > c2)
    
    aux = c2;
    c2  = c1;
    c1  = aux;
    
end

dl = (((a2-a1)*(b2-b1)*(c2-c1))/N)^(1/3); %%% Lattice constant

%%%% Generation of coordinates (solid parallelepiped)%%%%
x = -(a2-dl)/2:dl:(a2-dl)/2;
y = -(b2-dl)/2:dl:(b2-dl)/2;
z = -(c2-dl)/2:dl:(c2-dl)/2;


[XXX,YYY,ZZZ] = meshgrid(x,y,z);
X(:) = XXX(1:numel(XXX));
Y(:) = YYY(1:numel(YYY));
Z(:) = ZZZ(1:numel(ZZZ));

%%% Centering the distribution on the origin %%%
aux =  abs([min(X),max(X)]);
dX  =  max(aux-mean(aux));
if(max(X) < mean(aux))
    X = X+dX;
else
    X = X-dX;
end

aux =  abs([min(Z),max(Z)]);
dZ  =  max(aux-mean(aux));
if(max(Z) < mean(aux))
    Z = Z+dZ;
else
    Z = Z-dZ;
end

aux =  abs([min(Y),max(Y)]);
dY  =  max(aux-mean(aux));
if(max(Y) < mean(aux))
    Y = Y+dY;
else
    Y = Y-dY;
end

%%% Discarding inner positions of the shell %%%
k = 0;
for i = 1:length(X)
    
    condx = abs(X(i-k)) >= (a1+dl)/2; 
    condy = abs(Y(i-k)) >= (b1+dl)/2; 
    condz = abs(Z(i-k)) >= (c1+dl)/2;
    
    if(~(condx || condy || condz))
        X(i-k)  = [];
        Y(i-k)  = [];
        Z(i-k)  = [];
        k       = k+1;
    end
    
end

%%%% Displacing the center of mass %%%%
r(:,1) = X(:)+pos0(1);
r(:,2) = Y(:)+pos0(2);
r(:,3) = Z(:)+pos0(3);


%%%% Resulting number of dipoles %%%%
Nd = length(r(:,1));

end

